<?php
/**
 * Elementor Montoya Image Parallax Widget.
 *
 * Elementor widget that inserts a parallax image
 *
 * @since 1.0.0
 */
class Elementor_Montoya_Image_Parallax_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve image parallax widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'montoya_image_parallax';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve image parallax widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Image Parallax', 'montoya-elementor-widgets' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image parallax widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-image-rollover';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the image parallax widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'montoya-widgets' ];
	}

	/**
	 * Register image parallax widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Content', 'montoya-elementor-widgets' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'image',
			[
				'label' => __( 'Parallax Image', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				]
			]
		);

		$this->add_control(
			'video_url',
			[
				'label' => __( 'Video URL - MP4 format', 'montoya-elementor-widgets' ),
				'description' => __( 'For video parallax. Leave this field empty for parallax image effect.', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT
			]
		);
		
		$this->add_control(
			'content_width',
			[
				'label'        => __( 'Content Width', 'montoya-elementor-widgets' ),
				'type'         => Elementor\Controls_Manager::SELECT,
				'default'      => 'content-full-width',
				'options'      => array( 'content-full-width' => __( 'Full', 'montoya-elementor-widgets' ),
										'content-max-width' => __( 'Boxed', 'montoya-elementor-widgets' ) ),
				'prefix_class' => ''
			]
		);
		
		$this->add_control(
			'overlay_text',
			[
				'label' => __( 'Overlay Text', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::CODE,
				'language' => 'html',
				'rows' => 20,
			]
		);

		$this->add_control(
			'overlay_text_align',
			[
				'label'        => 'Overlay Text Alignment',
				'type'         => Elementor\Controls_Manager::SELECT,
				'default'      => 'text-align-center',
				'options'      => array( 'text-align-center' => __( 'Center', 'montoya-elementor-widgets' ),
										'text-align-left' => __( 'Left', 'montoya-elementor-widgets' ),
										'text-align-right' => __( 'Right', 'montoya-elementor-widgets' ) ),
				'prefix_class' => ''
			]
		);

		$this->add_control(
			'has_animation',
			[
				'label' => __( 'Has Animation', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => __( 'Yes', 'montoya-elementor-widgets' ),
				'label_off' => __( 'No', 'montoya-elementor-widgets' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'animation_delay',
			[
				'label' => __( 'Animation Delay', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'show_label' => true,
				'default' => 0
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Renders image parallax widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$settings = $this->get_settings_for_display();

		$montoya_animation = $settings['has_animation'];
		if( $montoya_animation == 'no'){

			$montoya_animation = false;
		}

		echo '<figure class="has-parallax has-parallax-content';
		if( $montoya_animation ){

			echo ' has-animation';
		}
		echo '"';
		if( $montoya_animation ){

			echo ' data-delay="'. esc_attr( $settings['animation_delay'] ) . '"';
		}
		echo '>';

		$montoya_elementor_img_alt = trim( strip_tags( get_post_meta( $settings['image']['id'], '_wp_attachment_image_alt', true ) ) );
		echo '<img src="' . esc_url( $settings['image']['url'] ) . '" alt="' . esc_attr( $montoya_elementor_img_alt ) . '">';
		if( !empty( $settings['video_url'] ) ){
		
			echo '<div class="content-video-wrapper">';
			echo '<video loop muted playsinline class="bgvid">';
			echo '<source src="' . esc_url( $settings['video_url'] ) . '" type="video/mp4">';
			echo '</video>';
			echo '</div>';
		}
		echo '<div class="parallax-image-content ' . esc_attr( $settings['content_width'] ) . ' ' . esc_attr( $settings['overlay_text_align'] ) . '">';
		echo '<div class="outer">';
		echo '<div class="inner">';
		echo wp_kses_post( $settings['overlay_text'] );
		echo '</div>';
		echo '</div>';
		echo '</div>';
		echo '</figure>';

	}

}

?>
