<?php
/**
 * Elementor News List Widget.
 *
 * Elementor widget that inserts a news list containing latest blog posts
 *
 * @since 1.0.0
 */
class Elementor_Montoya_News_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve News widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'montoya_news';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve News widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'News', 'montoya-elementor-widgets' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve News widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-archive-posts';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the News widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'montoya-widgets' ];
	}

	/**
	 * Register News widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Content', 'montoya-elementor-widgets' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'items_no',
			[
				'label' => __( 'Number of post items. Leave empty for ALL.', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'show_label' => true,
			]
		);

		$this->add_control(
			'filter_category',
			[
				'label' => __( 'Category filter. Add one or more blog categories separated by comma. Leave empty for ALL.', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'show_label' => true,
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render News widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$settings = $this->get_settings_for_display();

		$montoya_max_items		= $settings['items_no'];
		if( empty($montoya_max_items) ){

			$montoya_max_items = 1000;
		}
		

		$montoya_blog_tax_query = null;
		$montoya_blog_category_filter	= $settings['filter_category'];

		$montoya_portfolio_tax_query = null;

		$montoya_enable_ajax = false;
		if( function_exists( 'montoya_get_theme_options' ) ){

			$montoya_enable_ajax = montoya_get_theme_options( 'clapat_montoya_enable_ajax' );
		}

		$montoya_array_terms = null;
		if( !empty( $montoya_blog_category_filter ) ){

			$montoya_array_terms = explode( ",", $montoya_blog_category_filter );
			$montoya_portfolio_tax_query = array(
												array(
													'taxonomy' 	=> 'category',
													'field'		=> 'slug',
													'terms'		=> $montoya_array_terms,
													),
											);
		}

		echo '<div class="news-shortcode">';
		echo '<div id="blog-effects" class="' . sanitize_html_class( montoya_get_theme_options( 'clapat_montoya_blog_scroll_effect_type' ) ) . '" data-fx="1">';
		
		$montoya_paged = (get_query_var('paged')) ? get_query_var('paged') : 1;
		$montoya_args = array(
										'post_type' => 'post',
										'paged' => $montoya_paged,
										'tax_query' => $montoya_portfolio_tax_query,
										'posts_per_page' => $montoya_max_items,
										 );

		$montoya_portfolio = new WP_Query( $montoya_args );

		$montoya_current_item_count = 1;
		while( $montoya_portfolio->have_posts() ){

			$montoya_portfolio->the_post();
		
			$post_classes = get_post_class( 'post', get_the_ID() );
			
			$montoya_post_image = wp_get_attachment_image_src(get_post_thumbnail_id(get_the_ID()), 'full');

			$montoya_post_title = get_the_title();
			if( empty( $montoya_post_title ) ){
				
				$montoya_post_title = esc_html__("Post has no title", "montoya");
			}

			echo '<!-- Article -->';
			echo '<article id="post-'. get_the_ID() . '" class="news-post ' . esc_attr( implode( ' ', $post_classes ) ) . '">';
			echo '<div class="article-bg content-full-width">';
								
			echo '<div class="article-content content-full-width">';

			// Post categories
			echo '<div class="entry-meta-wrap">';
			echo '<div class="entry-meta entry-categories">';
			echo '<ul class="post-categories">';
			$montoya_categories = get_the_category();
			if ( ! empty( $montoya_categories ) ) {

				foreach( $montoya_categories as $montoya_category ) {

					echo '<li class="link">';
					echo wp_kses( '<a class="ajax-link" data-type="page-transition" href="' . esc_url( get_category_link( $montoya_category->term_id ) ) . '" rel="category tag"><span data-hover="' . esc_attr( $montoya_category->name ) . '">' . esc_html( $montoya_category->name ) . '</span></a>', 'montoya_allowed_html' );
					echo '</li>';
				}
			}
			echo '</ul>';
			echo '</div>';
			echo '</div>';

			// Post date
			echo '<div class="entry-meta-wrap">';
			echo '<ul class="entry-meta entry-date">';
			echo '<li class="link"><a class="ajax-link" href="' . get_permalink() . '"><span data-hover="' . get_the_date() . '">' . get_the_date() . '</span></a></li>';
			echo '</ul>';
			echo '</div>';
								
			echo '</div>';

			echo '<div class="article-wrap">';
								
			// Post image
			echo '<div class="article-show-image">';
			if( $montoya_post_image  ){
					
				echo '<div class="hover-reveal">';
				echo '<div class="hover-reveal__inner">';
				echo '<div class="hover-reveal__img">';
				echo '<a class="ajax-link" href="' . esc_url( get_permalink() ) . '" data-type="page-transition">';
				echo '<img src="' . esc_url( $montoya_post_image[0] ) . '" alt="' . esc_attr__("Post Image", "montoya") . '"';
				echo '</a>';
				echo '</div>';
				echo '</div>';
				echo '</div>';
			} // if post image
									
			// Post link
			echo '<a class="post-title ajax-link hide-ball" href="' . esc_url( get_permalink() ) . '" data-type="page-transition">' . wp_kses( $montoya_post_title, 'montoya_allowed_html' ) . '</a>';
								
			echo '</div>';

			echo '<div class="article-links">';
			echo '<div class="button-box">';
			echo '<a class="ajax-link" data-type="page-transition" href="' . get_permalink() . '">';
			echo '<div class="clapat-button-wrap parallax-wrap hide-ball">';
			echo '<div class="clapat-button parallax-element">';
			echo '<div class="button-border outline rounded parallax-element-second">';
			echo '<span data-hover="' . esc_attr__('Read Article', 'montoya') . '">' . esc_html__('Read Article', 'montoya') . '</span>';
			echo '</div>';
			echo '</div>';
			echo '</div>';
			echo '</a>';
			echo '</div>';
			echo '</div>';

			// Post Read More button
			echo '<div class="button-box">';
			echo '<a class="ajax-link" href="' . esc_url( get_permalink() ) . '" data-type="page-transition">';
			echo '<div class="clapat-button-wrap parallax-wrap hide-ball">';
			echo '<div class="clapat-button parallax-element">';
			echo '<div class="button-border outline rounded parallax-element-second">';
			echo '<span data-hover="' . esc_attr( montoya_get_theme_options( 'clapat_montoya_blog_read_more_caption' ) ) . '">' . wp_kses( montoya_get_theme_options( 'clapat_montoya_blog_read_more_caption' ), 'montoya_allowed_html' ) . '</span>';
			echo '</div>';
			echo '</div>';
			echo '</div>';
			echo '</a>';
			echo '</div>';
			

			echo '</div>';

			echo '</div>';
			echo '</article>';
			echo '<!--/Article -->';
					
			$montoya_current_item_count++;

		}

		wp_reset_postdata();
		
		echo '</div>';
		echo '</div>';

	}

}

?>
