<?php
/**
 * Elementor Montoya Portfolio Grid Widget.
 *
 * Elementor widget that inserts a portfolio grid containing the thumbnails of portfolio items defined in the system
 *
 * @since 1.0.0
 */
class Elementor_Montoya_Portfolio_Grid_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Portfolio Grid widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'montoya_portfolio_grid';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Portfolio Grid widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Portfolio Grid', 'montoya-elementor-widgets' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Portfolio Grid widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-posts-masonry';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Portfolio Grid widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'montoya-widgets' ];
	}

	/**
	 * Register Portfolio Grid widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Content', 'montoya-elementor-widgets' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'items_no',
			[
				'label' => __( 'Number of portfolio items. Maximum of 4 items can be included. First 4 if you leave it empty.', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'show_label' => true,
			]
		);

		$this->add_control(
			'filter_category',
			[
				'label' => __( 'Category filter. Add one or more portfolio categories separated by comma. First 4 if you leave it empty.', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'show_label' => true,
			]
		);
		
		$this->add_control(
			'thumbs_effect',
			[
				'label'        => __( 'Thumbs Transition Effect', 'montoya-elementor-widgets' ),
				'type'         => \Elementor\Controls_Manager::SELECT,
				'label_block'  => true,
				'default'      => 'webgl-fitthumbs',
				'options'      => array( 'webgl-fitthumbs' => __( 'WebGL Animation', 'montoya-elementor-widgets' ),
										'no-fitthumbs' => __( 'None', 'montoya-elementor-widgets' )
										)
			]
		);

		$this->add_control(
			'thumbs_effect_webgl',
			[
				'label'        => __( 'WebGL Animation Type', 'montoya-elementor-widgets' ),
				'type'         => \Elementor\Controls_Manager::SELECT,
				'label_block'  => true,
				'default'      => 'fx-one',
				'options'      => array( 'fx-one' => __( 'FX One', 'montoya-elementor-widgets' ),
										'fx-two' => __( 'FX Two', 'montoya-elementor-widgets' ),
										'fx-three' => __( 'FX Three', 'montoya-elementor-widgets' ),
										'fx-four' => __( 'FX Four', 'montoya-elementor-widgets' ),
										'fx-five' => __( 'FX Five', 'montoya-elementor-widgets' ),
										'fx-six' => __( 'FX Six', 'montoya-elementor-widgets' ),
										'fx-seven' => __( 'FX Seven', 'montoya-elementor-widgets' ) )
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render Portfolio Grid widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$settings = $this->get_settings_for_display();

		$montoya_portfolio_tax_query = null;
		$montoya_portfolio_category_filter = $settings['filter_category'];

		$montoya_portfolio_thumb_to_fullscreen = $settings['thumbs_effect'];
		$montoya_portfolio_thumb_webgl = $settings['thumbs_effect_webgl'];
		$montoya_enable_ajax = false;
		if( function_exists( 'montoya_get_theme_options' ) ){
			
			$montoya_enable_ajax = montoya_get_theme_options( 'clapat_montoya_enable_ajax' );
		}
			
		if( !$montoya_enable_ajax ){
			
			$montoya_portfolio_thumb_to_fullscreen = 'no-fitthumbs';
		}

		$montoya_array_terms = null;
		if( !empty( $montoya_portfolio_category_filter ) ){

			$montoya_array_terms = explode( ",", $montoya_portfolio_category_filter );
			$montoya_portfolio_tax_query = array(
												array(
													'taxonomy' 	=> 'portfolio_category',
													'field'		=> 'slug',
													'terms'		=> $montoya_array_terms,
													),
											);
		}

		// add the wrapper links divs only if they don't exist already in the page, otherwise it will inherit the existing divs
		if( !is_page_template('portfolio-showcase-grid-page.php') ){
		
			echo '<!-- Showcase List Holder -->';
			echo '<div id="itemsWrapperLinks">';
			echo '<!-- Showcase List -->';
			echo '<div id="itemsWrapper" class="' .  sanitize_html_class( $montoya_portfolio_thumb_to_fullscreen ) . ' ' . sanitize_html_class( $montoya_portfolio_thumb_webgl ) . '">';
		}
		
		echo '<!-- ClaPat Portfolio -->';
		echo '<div class="showcase-portfolio">';
		
		$montoya_paged = (get_query_var('paged')) ? get_query_var('paged') : 1;
		$montoya_items_no = 1000;
		if( !empty( $settings['items_no'] ) ){
		
			$montoya_items_no = $settings['items_no'];
		}
		
		$montoya_args = array(
								'post_type' => 'montoya_portfolio',
								'paged' => $montoya_paged,
								'tax_query' => $montoya_portfolio_tax_query,
								'posts_per_page' => $montoya_items_no
							 );

		$montoya_portfolio = new WP_Query( $montoya_args );

		$montoya_current_item_count = 1;
		while( $montoya_portfolio->have_posts() ){

			$montoya_portfolio->the_post();
			
			$full_image = montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-hero-img' );
			$montoya_background_type = montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-bknd-color' );
			$montoya_background_color = montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-bknd-color-code' );
			$title_row = montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-hero-caption-title' );
			$title_list				= preg_split('/\r\n|\r|\n/', $title_row);
			$montoya_item_title		= "";
			foreach( $title_list as $title_bit ){

				$montoya_item_title .= '<span>' . $title_bit . '</span>';
			}
			$subtitle_row 		= montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-hero-caption-subtitle' );
			$subtitle_list			= preg_split('/\r\n|\r|\n/', $subtitle_row);
			$montoya_item_subtitle	= "";
			foreach( $subtitle_list as $subtitle_bit ){

				$montoya_item_subtitle .= '<span>' . $subtitle_bit . '</span>';
			}
			$montoya_background_navigation = montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-navigation-cursor-color' );
			$montoya_page_caption_first_line	= montoya_get_theme_options( 'clapat_montoya_view_project_caption_first' );
			$montoya_page_caption_second_line	= montoya_get_theme_options( 'clapat_montoya_view_project_caption_second' );
			$montoya_project_year	=  montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-project-year' );
			
			$montoya_curtain_color = montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-curtain-color-code' );
			
			if( $full_image && isset( $full_image['url'] ) ){

				$montoya_item_classes = '';
						
				$montoya_item_categories 	= '';
				$montoya_item_cats = get_the_terms( get_the_ID(), 'portfolio_category' );
				if($montoya_item_cats){

					foreach($montoya_item_cats as $item_cat) {
						
						$montoya_item_classes 	.= $item_cat->slug . ' ';
						$montoya_item_categories 	.= $item_cat->name . ', ';
					}

					$montoya_item_categories = rtrim($montoya_item_categories, ', ');

				}
				
				$montoya_image_alt_text = __('Portfolio Image', 'montoya_core_plugin_text_domain');
				if( !empty( $full_image['id'] ) ){
					
					$montoya_image_alt_text = trim( strip_tags( get_post_meta( $full_image['id'], '_wp_attachment_image_alt', true ) ) );
					
				}
				
				$montoya_change_header = "";
				if( $montoya_background_type  == 'dark-content' ){
										
					$montoya_change_header = "change-header";
				}
	
				$item_url = get_the_permalink();
				
				echo '<div class="clapat-item ';
				if( ($montoya_current_item_count % 6 == 2) || ($montoya_current_item_count % 6 == 4) ){
	
					echo 'vertical-parallax ';
				}
				echo esc_attr( $montoya_item_classes ) . '">';
				echo '<div class="slide-inner trigger-item ' . sanitize_html_class( $montoya_change_header ) . '" data-centerLine="' . esc_attr( montoya_get_theme_options('clapat_montoya_open_project_caption') ) . '">';
				echo '<div class="img-mask">';
				echo '<div class="curtains" data-curtain-color="' . esc_attr( $montoya_curtain_color ) . '">';
				echo '<div class="curtain-row"></div>';
				echo '</div>';
				echo '<a class="slide-link" data-type="page-transition" href="' . esc_url( $item_url ) . '"></a>';
				echo '<div class="section-image trigger-item-link">';
				echo '<img src="' . esc_url( $full_image['url'] ) . '" class="item-image grid__item-img" alt="' . esc_attr( $montoya_image_alt_text ) . '">';
				if( montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-video' ) ){

					$montoya_video_webm_url = montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-video-webm' );
					$montoya_video_mp4_url 	= montoya_get_post_meta( MONTOYA_THEME_OPTIONS, get_the_ID(), 'montoya-opt-portfolio-video-mp4' );
					
					echo '<div class="hero-video-wrapper">';
					echo '<video loop muted class="bgvid">';
					if( !empty( $montoya_video_mp4_url ) ) {
						
						echo '<source src="' . esc_url( $montoya_video_mp4_url ) . '" type="video/mp4">';
					}
					if( !empty( $montoya_video_webm_url ) ) {
						
						echo '<source src="' . esc_url( $montoya_video_webm_url ) . '" type="video/webm">';
					}
					echo '</video>';
					echo '</div>';
				}
				echo '</div>';
				echo '<img class="grid__item-img grid__item-img--large" src="' . esc_url( $full_image['url'] ) . '" alt="' . esc_attr( $montoya_image_alt_text ) . '" />';
				echo '</div>';
				echo '<div class="slide-caption trigger-item-link-secondary">';
				echo '<div class="slide-title">' . wp_kses( $montoya_item_title, 'montoya_allowed_html' ) . '</div>';
				echo '<div class="slide-date"><span>' . wp_kses( $montoya_project_year, 'montoya_allowed_html' ) . '</span></div>';
				echo '<div class="slide-cat"><span>' . wp_kses( $montoya_item_categories, 'montoya_allowed_html' ) . '</span></div>';
				echo '</div>';
				echo '</div>';
				echo '</div>';
				
			}
			
			$montoya_current_item_count++;

		}

		wp_reset_postdata();
 		
		echo '</div>';
		echo '<!--/Clapat Portfolio-->';
			
		if( !is_page_template('portfolio-showcase-grid-page.php') ){
			
			echo '</div>';
			echo '<!--/Showcase List-->';
			echo '</div>';
			echo '<!--/Showcase List Holder-->';
		}

	}

}
 
?>
